<?php

/**
 * @package         Google Structured Data Markup
 * @version         2.1.5 Free
 *
 * @author          Tassos Marinos <info@tassos.gr>
 * @link            http://www.tassos.gr
 * @copyright       Copyright © 2016 Tassos Marinos All Rights Reserved
 * @license         GNU GPLv3 <http://www.gnu.org/licenses/gpl.html> or later
 */

defined('_JEXEC') or die('Restricted Access');

/**
 *  Google Structured Data JSON generator
 */
class GSDJSON
{
	/**
	 *  Snippet Data
	 *
	 *  @var  object
	 */
	private $data;

	/**
	 *  Class Constructor
	 *
	 *  @param  object  $data     The snippet object
	 */
	public function __construct($data = null)
	{
		$this->setData($data);
	}

	/**
	 *  Set Data
	 *
	 *  @param  array  $data
	 */
	public function setData($data)
	{
		if (!is_array($data))
		{
			return;
		}

		$this->data = new JRegistry($data);
		return $this;
	}

	/**
	 *  Get Snippet JSON
	 *
	 *  @return  string
	 */
	public function generate()
	{
		$snippetMethod = "snippet" . $this->data->get("snippet");

		if (!method_exists($this, $snippetMethod))
		{
			return;
		}

		// Call Snippet
		if (!$result = $this->$snippetMethod())
		{
			return;
		}

		$json = '
            <script type="application/ld+json">
            {
                ' . implode(",", $result) . '
            }
            </script>';

		// Add Custom Code
		$customCode = $this->data->get("custom", null);
		if ($customCode && !empty($customCode) && strpos($customCode, '</script>') !== false)
		{
			$json .= "\n" . $customCode . "\n";
		}

		return $json;
	}

	/**
	 *  Generates Product Rich Snippet
	 *  https://developers.google.com/search/docs/data-types/products
	 *
	 *  @return  string
	 */
	private function snippetProduct()
	{
		$json[] = '"@context": "https://schema.org/",
                "@type": "Product",
                "name": "' . $this->data->get("title") . '",
                "image": "' . $this->data->get("image") . '",
                "description": "' . $this->data->get("description") . '"';

		// Brand
		if ($this->data->get("brand"))
		{
			$json[] = '
                "brand": {
                    "@type": "Thing",
                    "name": "' . $this->data->get("brand") . '"
                }';
		}

		// SKU
		if ($this->data->get("sku"))
		{
			$json[] = '
                "sku": "' . $this->data->get("sku") . '"';
		}

		// Aggregate Rating
		if ($this->data->get("ratingValue") && $this->data->get("reviewCount"))
		{
			$json[] = '
                "aggregateRating": {
                    "@type": "AggregateRating",
                    "ratingValue": "' . $this->data->get("ratingValue") . '",
                    "reviewCount": "' . $this->data->get("reviewCount") . '",
                    "worstRating": "' . $this->data->get("worstRating", 0) . '",
                    "bestRating": "' . $this->data->get("bestRating", 5) . '"
                }';
		}

		// Offer / Pricing
		$json[] = '
                "offers": {
                    "@type": "Offer",
                    "priceCurrency": "' . $this->data->get("currency", "USD") . '",
                    "price": "' . $this->data->get("price", "0.00") . '",
                    "url": "' . $this->data->get("url") . '",
                    "itemCondition": "' . $this->data->get("condition", "http://schema.org/NewCondition") . '",
                    "availability": "' . $this->data->get("availability", "http://schema.org/InStock") . '"
                }';

		return $json;
	}

	/**
	 *  Generates Article Rich Snippet
	 *
	 *  @return  string
	 */
	private function snippetArticle()
	{
		$json[] = '"@context": "https://schema.org/",
                "@type": "Article",
                "mainEntityOfPage": {
                    "@type": "WebPage",
                    "@id": "' . $this->data->get("url") . '"
                },
                "headline": "' . $this->data->get("title") . '",
                "description": "' . $this->data->get("description") . '",
                "image": {
                    "@type": "ImageObject",
                    "url": "' . $this->data->get("image") . '",
                    "height": 800,
                    "width": 800
                }';

		// Aggregate Rating
		if ($this->data->get("ratingValue") && $this->data->get("reviewCount"))
		{
			$json[] = '
                "aggregateRating": {
                    "@type": "AggregateRating",
                    "ratingValue": "' . $this->data->get("ratingValue") . '",
                    "reviewCount": "' . $this->data->get("reviewCount") . '",
                    "worstRating": 0,
                    "bestRating": 5
                }';
		}

		// Author
		if ($this->data->get("author.name"))
		{
			$json[] = '
                "author": {
                    "@type": "Person",
                    "name": "' . $this->data->get("author.name") . '"
                }';
		}

		// Publisher
		if ($this->data->get("publisher.name"))
		{
			$json[] = '
                "publisher": {
                    "@type": "Organization",
                    "name": "' . $this->data->get("publisher.name") . '",
                    "logo": {
                        "@type": "ImageObject",
                        "url": "' . $this->data->get("publisher.logo") . '",
                        "width": 600,
                        "height": 60
                    }
                }';
		}

		$json[] = '
                "datePublished": "' . $this->data->get("created") . '",
                "dateModified": "' . $this->data->get("modified") . '"';

		return $json;
	}

	/**
	 *  Generates Events Rich Snippet
	 *
	 *  https://developers.google.com/search/docs/data-types/events
	 *
	 *  @return  string
	 */
	private function snippetEvent()
	{
		$json[] = '"@context": "https://schema.org",
                "@type": "' . $this->data->get("type") . '",
                "name": "' . $this->data->get("title") . '",
                "image": "' . $this->data->get("image") . '",
                "description": "' . $this->data->get("description") . '",
                "url": "' . $this->data->get("url") . '",
                "startDate": "' . $this->data->get("startdate") . '",
                "doorTime": "' . $this->data->get("starttime") . '",
                "endDate": "' . $this->data->get("enddate") . '",
                "eventStatus": "' . $this->data->get("status") . '",
                "location": {
                    "@type": "EventVenue",
                    "name": "' . $this->data->get("location.name") . '",
                    "address": "' . $this->data->get("location.address") . '"
                }';

		if ($this->data->get("performer.name"))
		{
			$json[] ='
                "performer": {
                    "@type": "' . $this->data->get("performer.type") . '",
                    "name": "' . $this->data->get("performer.name") . '"
                }';
		}

		$json[] =
		       '
                "offers": {
                    "@type": "Offer",
                    "category": "primary",
                    "url": "' . $this->data->get("url") . '",
                    "availability": "' . $this->data->get("offer.availability") . '",
                    "availabilityStarts": "' . $this->data->get("offer.startDateTime") . '",
                    "price": "' . $this->data->get("offer.price") . '",
                    "priceCurrency": "' . $this->data->get("offer.currency") . '",
                    "inventoryLevel": {
                        "@context": "https://schema.org",
                        "@type": "QuantitativeValue",
                        "value": "' . $this->data->get("offer.inventoryLevel") . '",
                        "unitText": "Tickets"
                    }
                }';

		return $json;
	}

	/**
	 *  Returns Breadcrumbs Rich Snippet
	 *
	 *  @return  string
	 */
	private function snippetBreadcrumbs()
	{
		$crumbsData = array();
        $crumbs = $this->data->get("crumbs");
        
        if (!is_array($crumbs))
        {
            return;
        }

		foreach ($crumbs as $key => $value)
		{
			$crumbsData[] = '{
                    "@type": "ListItem",
                    "position": ' . ($key + 1) . ',
                    "item": {
                        "@id":  "' . $value->link . '",
                        "name": "' . $value->name . '"
                    }
                }';
		}

		$json[] = '"@context": "https://schema.org",
                "@type": "BreadcrumbList",
                "itemListElement": [' . implode(",", $crumbsData) . ']';

		return $json;
	}

	/**
	 *  Returns Site Name Rich Snippet
	 *  https://developers.google.com/structured-data/site-name
	 *
	 *  @return  string on success, boolean on fail
	 */
	private function snippetSiteName()
	{
		$json[] = '"@context": "https://schema.org",
                "@type": "WebSite",
                "name": "' . $this->data->get("name") . '",
                "url": "' . $this->data->get("url") . '"';

		if ($this->data->get("alt"))
		{
			$json[] = '
                "alternateName": "' . $this->data->get("alt") . '"';
		}

		return $json;
	}

	/**
	 *  Returns Sitelinks Searchbox Rich Snippet
	 *  https://developers.google.com/search/docs/data-types/sitelinks-searchbox
	 *
	 *  @return  string on success, boolean on fail
	 */
	private function snippetSearch()
	{
		$json[] = '"@context": "https://schema.org",
                "@type": "WebSite",
                "url": "' . $this->data->get("siteurl") . '",
                "potentialAction": {
                    "@type": "SearchAction",
                    "target": "' . $this->data->get("searchurl") . '",
                    "query-input": "required name=search_term"
                }';

		return $json;
	}

	/**
	 *  Returns Site Logo Rich Snippet
	 *  https://developers.google.com/search/docs/data-types/logo
	 *
	 *  @return  string on success, boolean on fail
	 */
	private function snippetLogo()
	{
		$json[] = '"@context": "https://schema.org",
                "@type": "Organization",
                "url": "' . $this->data->get("url") . '",
                "logo": "' . $this->data->get("logo") . '"';

		return $json;
	}

	/**
	 *  Returns Social Profiles Rich Snippet
	 *  https://developers.google.com/search/docs/data-types/social-profile-links
	 *
	 *  @return  string on success, boolean on fail
	 */
	private function snippetSocialProfiles()
	{
		// JSON
		$json[] = '"@context": "https://schema.org",
                "@type": "' . $this->data->get("type") . '",
                "name": "' . $this->data->get("sitename") . '",
                "url": "' . $this->data->get("siteurl") . '",
                "sameAs": [
                  "' . implode("\",\n\"", (array) $this->data->get("links")) . '"
                ]';

		return $json;
	}

    /**
     *  Returns Business Listing Snippet
     *  https://developers.google.com/search/docs/data-types/local-businesses
     *
     *  @return  string on success, boolean on fail
     */
    private function snippetBusinessListing()
    {
        $openingHours = '';
        $weekDays = $this->data->get('weekDays');

        switch ($this->data->get('hoursAvailable', '0'))
        {
            case '0':
                $openingHours = '"
                openingHoursSpecification": []';
                break;
            case '1':
                $openingHours = '
                "openingHoursSpecification": [{
                    "@type": "OpeningHoursSpecification",
                    "dayOfWeek": [
                        "Monday",
                        "Tuesday",
                        "Wednesday",
                        "Thursday",
                        "Friday",
                        "Saturday",
                        "Sunday"
                    ],
                    "opens": "00:00",
                    "closes": "23:59"
                }]';
                break;
            case '2':
                $openingHours = '
                "openingHoursSpecification": [';

                foreach ($weekDays as $weekDay)
                {
                    $dayEnabled      = $this->data->get($weekDay);
                    $dayEnabledStart = $this->data->get($weekDay . '_start');
                    $dayEnabledEnd   = $this->data->get($weekDay . '_end');

                    if ($dayEnabled && $dayEnabledStart && $dayEnabledEnd)
                    {
                        $openingHours .= '{
                            "@type": "OpeningHoursSpecification",
                            "dayOfWeek": [
                                "' . $weekDay . '"
                            ],
                            "opens": "' . $dayEnabledStart . '",
                            "closes": "' . $dayEnabledEnd . '"
                        },';
                    }
                }

                $openingHours = (substr($openingHours, -1) == ',') ? rtrim($openingHours, ",") : $openingHours;
                $openingHours .= ']';
                break;
        }
        
        $coordinates = $this->data->get('coordinates');

        // JSON
        $json[] = '"@context": "https://schema.org",
                "@type": "' . $this->data->get('type') . '",
                "image": "' . $this->data->get('image') . '",
                "@id": "' . $this->data->get('id') . '",
                "name": "' . $this->data->get('name') . '",
                "priceRange": "' . $this->data->get('price_range') . '",
                "address": {
                    "@type": "PostalAddress",
                    "streetAddress": "' . $this->data->get('streetAddress') . '",
                    "addressLocality": "' . $this->data->get('addressLocality') . '",
                    "addressRegion": "' . $this->data->get('addressRegion') . '",
                    "postalCode": "' . $this->data->get('postalCode') . '",
                    "addressCountry": "' . $this->data->get('addressCountry') . '"
                },
                "geo": {
                    "@type": "GeoCoordinates",
                    "latitude": ' . $coordinates[0] . ',
                    "longitude": ' . $coordinates[1] . '
                },
                "url": "' .$this->data->get('id') . '",
                "telephone": "' . $this->data->get('telephone') . '",'
                . $openingHours . '';

        return $json;
    }

	/**
	 *  Returns Recipes Rich Snippet
	 *  https://developers.google.com/search/docs/data-types/recipes
	 *
	 *  @return  string on success, boolean on fail
	 */
	private function snippetRecipe()
	{
		// JSON
		$json[] = '"@context": "https://schema.org",
                "@type": "Recipe",
                "name": "' . $this->data->get("title") . '",
                "image": "' . $this->data->get("image") . '"';

		// Author Data
		if ($this->data->get("authorName"))
		{
			$json[] = '
                "author": {
                    "@type": "Person",
                    "name": "' . $this->data->get("authorName") . '"
                }';
		}

		$json[] = '
                "datePublished" : "' . $this->data->get("datePublished") . '",
                "dateCreated" : "' . $this->data->get("dateCreated") . '",
                "dateModified" : "' . $this->data->get("dateModified") . '",
                "description": "' . $this->data->get("description") . '"';

		// Aggregate Rating
		if ($this->data->get("ratingValue") && $this->data->get("reviewCount"))
		{
			$json[] = '
                "aggregateRating": {
                    "@type": "AggregateRating",
                    "ratingValue": "' . $this->data->get("ratingValue") . '",
                    "reviewCount": "' . $this->data->get("reviewCount") . '",
                    "worstRating": 0,
                    "bestRating": 5
                }';
		}

		$json[] = '
                "prepTime": "' . $this->data->get("prepTime") . '",
                "cookTime": "' . $this->data->get("cookTime") . '",
                "totalTime": "' . $this->data->get("totalTime") . '",
                "nutrition": {
                    "@type": "NutritionInformation",
                    "calories": "' . $this->data->get("calories") . '"
                },
                "recipeYield": "' . $this->data->get("yield") . '",
                "recipeIngredient": ' . json_encode($this->data->get("ingredient")) . ',
                "recipeInstructions": ' . json_encode($this->data->get("instructions"));

		return $json;
	}

    /**
     *  Returns Courses Rich Snippet
     *  https://developers.google.com/search/docs/data-types/courses
     *
     *  @return  string on success, boolean on fail
     */
	private function snippetCourse()
	{
		$json[] = '"@context": "https://schema.org",
                "@type": "Course",
                "name": "' . $this->data->get("title") . '",
                "description": "' . $this->data->get("description") . '",
                "provider": {
                    "@type": "Organization",
                    "name": "' . $this->data->get("sitename") . '",
                    "sameAs": "' . $this->data->get("siteurl") . '"
                }';

        return $json;
	}

    /**
     *  Returns Review Rich Snippet
     *  https://developers.google.com/search/docs/data-types/reviews
     *
     *  @return  string on success, boolean on fail
     */
    private function snippetReview()
    {
        $json[] = '"@context": "https://schema.org",
                "@type": "Review",
                "author": {
                    "@type": "Person",
                    "name": "' . $this->data->get("authorName") . '",
                    "sameAs": "' . $this->data->get("siteurl") . '"
                },
                "url": "' . $this->data->get("url") . '",
                "datePublished": "' . $this->data->get("datePublished") . '",
                "publisher": {
                    "@type": "Organization",
                    "name": "' . $this->data->get("sitename") . '",
                    "sameAs": "' . $this->data->get("siteurl") . '"
                },
                "description": "' . $this->data->get("description") . '",
                "itemReviewed": {
                    "@type": "' . $this->data->get("itemReviewedType") . '",
                    "image": "' . $this->data->get("image") . '"';
        
        if ($this->data->get("itemReviewedType") == "LocalBusiness") 
        {
            $json[] = '
                    "address": {
                        "@type": "PostalAddress",
                        "name": "' . $this->data->get("address") . '"
                     },
                    "priceRange": "' . $this->data->get("priceRange") . '",
                    "telephone": "' . $this->data->get("telephone") . '"';
        }

        $json[] = '
                    "name": "' . $this->data->get("itemReviewedName") . '"
                },
                "reviewRating": {
                    "@type": "Rating",
                    "ratingValue": ' . $this->data->get("ratingValue") . ',
                    "worstRating": 0,
                    "bestRating": 5
                }';

        return $json;
    }
}

?>