<?php

/**
 * @package         Google Structured Data Markup
 * @version         2.1.5 Free
 *
 * @author          Tassos Marinos <info@tassos.gr>
 * @link            http://www.tassos.gr
 * @copyright       Copyright © 2016 Tassos Marinos All Rights Reserved
 * @license         GNU GPLv3 <http://www.gnu.org/licenses/gpl.html> or later
 */

defined('_JEXEC') or die('Restricted Access');

/**
 *  Google Structured Data Helper Class
 */
class GSDHelper
{
	/**
	 *  Plugin Params
	 *
	 *  @var  JRegistry
	 */
	public static $params;

	/**
	 *  Generates JSON by active component
	 *
	 *  @return  mixed  Null on fail, String on success
	 */
	public static function getJSONByComponent()
	{
		$componentName = self::getComponentName();

		// Check if we have support the component with an integration
		if (!in_array($componentName, self::getIntegrationList()))
		{
			self::log($componentName . " integration not found");
			return;
		}

		// Load Integration Class
		if (!$class = self::getIntegrationClass($componentName))
		{
			return;
		}

		// Validate Integration Data
		$data = $class->getData();
		if (!is_array($data) || !isset($data["snippet"]))
		{
			return;
		}

		self::log($data);
		
		// Generate JSON
		$json = new GSDJSON($data);
		$script = $json->generate();

		self::log($script);	

		return $script;
	}

	/**
	 *  Get Integration Class
	 *
	 *  @param   string  $name    Integration Name
	 *
	 *  @return  object           Integration Class
	 */
	public static function getIntegrationClass($name)
	{
		if (!@include_once(self::getIntegrationsPath() . $name . "/" . $name . ".php"))
        {
            return;
        }

		$class = "GSD" . $name;

		return new $class(GSDHelper::getParams());
	}

	/**
	 *  Loads integrations form fields to plugin editing page
	 *
	 *  @return  void
	 */
	public static function loadIntegrationsForms(&$form)
	{
		foreach (self::getIntegrationList() as $integration)
		{
			JForm::addFormPath(self::getIntegrationsPath() . $integration);
			$form->loadFile($integration, false);

			if (!NRFrameworkFunctions::extensionInstalled($integration))
			{
				$form->setFieldAttribute("enabled", "disabled", true, "params." . $integration);
			}
		}	
	}

	public static function getIntegrationList()
	{
		return JFolder::folders(self::getIntegrationsPath());
	}

	public static function getIntegrationsPath()
	{
		return __DIR__ . "/integrations/";
	}

	/**
	 *  Returns an array with crumbs
	 *
	 *  @return  array
	 */
	public static function getCrumbs($hometext)
	{
		$pathway = JFactory::getApplication()->getPathway();
		$items   = $pathway->getPathWay();
		$menu    = JFactory::getApplication()->getMenu();
		$lang    = JFactory::getLanguage();
		$count   = count($items);

		// Look for the home menu
		if (JLanguageMultilang::isEnabled())
		{
			$home = $menu->getDefault($lang->getTag());
		}
		else
		{
			$home = $menu->getDefault();
		}

		if (!$count)
		{
			return false;
		}

		// We don't use $items here as it references JPathway properties directly
		$crumbs = array();

		for ($i = 0; $i < $count; $i++)
		{
			$crumbs[$i]       = new stdClass;
			$crumbs[$i]->name = stripslashes(htmlspecialchars($items[$i]->name, ENT_COMPAT, 'UTF-8'));
			$crumbs[$i]->link = self::route($items[$i]->link);
		}

		// Add Home item
		$item       = new stdClass;
		$item->name = htmlspecialchars($hometext);
		$item->link = self::route('index.php?Itemid=' . $home->id);
		array_unshift($crumbs, $item);

		return $crumbs;
	}

	/**
	 *  Returns image width and height
	 *
	 *  @param   string  $image  The URL of the image2wbmp(image)
	 *
	 *  @return  array
	 */
	public static function getImageSize($image)
	{
		if (!ini_get('allow_url_fopen') || !function_exists('getimagesize'))
		{
			return array("width" => 0, "height" => 0);
		}

		$imageSize = $image ? getimagesize($image) : array(0, 0);

		$info["width"]  = $imageSize[0];
		$info["height"] = $imageSize[1];

		return $info;
	}

	/**
	 *  Makes text safe for JSON outpout
	 *
	 *  @param   string   $text   The text 
	 *  @param   integer  $limit  Limit characters
	 *
	 *  @return  string
	 */
	public static function makeTextSafe($text, $limit = 200)
	{
		$text = trim(strip_tags($text));

		if ($limit > 0)
		{
			$text = mb_substr($text, 0, $limit);
		}

		// Escape
        $text = addcslashes($text, '"\\/');

        return $text;
	}

	/**
	 *  Returns absolute URL
	 *
	 *  @param   string  $url  The URL
	 *
	 *  @return  string
	 */
	public static function absURL($url)
	{
		$url = str_replace(JURI::root(), "", $url);
		$url = str_replace(JURI::root(true), "", $url);
		$url = ltrim($url, "/");
		
		return JURI::root() . $url;
	}

	/**
	 *  Returns URLs based on the Force SSL global configuration
	 *
	 *  @param   string   $route  The route for which we want a URL
	 *  @param   boolean  $xhtml  If we want the output to be in XHTML
	 *
	 *  @return  string           The absolute url
	 */
	public static function route($route, $xhtml = true)
	{
		$siteSSL = JFactory::getConfig()->get('force_ssl');
		$sslFlag = 2;

		// the force_ssl value in the global configuration needs
		// to be 2 for the frontend to also be under HTTPS
		if (($siteSSL == 2) || (isset($_SERVER['HTTPS']) && !empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off'))
		{
			$sslFlag = 1;
		}
		
		return JRoute::_($route, $xhtml, $sslFlag);
	}

	/**
	 *  Formats date based on ISO8601
	 *
	 *  @param   JDate  $date  
	 *
	 *  @return  JDate
	 */
	public static function date($date)
	{
		return JFactory::getDate($date)->toISO8601();
	}

	/**
	 *  Get current active component name
	 *
	 *  @return  string
	 */
	public static function getComponentName()
	{
		if (!$option = JFactory::getApplication()->input->get('option', null))
		{
			return false;
		}

		$optionParts   = explode("_", $option);
		$componentName = isset($optionParts[1]) ? $optionParts[1] : null;

		if (!$componentName)
		{
			return false;
		}

		return $componentName;
	}

	/**
	 *  Determine if the user is viewing the front page
	 *
	 *  @return  boolean
	 */
	public static function isFrontPage()
	{
		$menu = JFactory::getApplication()->getMenu();
		$lang = JFactory::getLanguage()->getTag();
		return ($menu->getActive() == $menu->getDefault($lang));
	}

    /**
     *  Logs messages to log file
     *
     *  @param   string  $msg   The message
     *  @param   object  $type  The log type
     *
     *  @return  void
     */
    public static function log($msg)
    {
        if (JDEBUG)
        {
        	var_dump($msg);
        }
    }

    /**
     *  Get website name
     *
     *  @return  string  Site URL
     */
    public static function getSiteName()
    {
        return self::getParams()->get("sitename_name", JFactory::getConfig()->get('sitename'));
    }
    
    /**
     *  Returns the Site Logo URL
     *
     *  @return  string
     */
    public static function getSiteLogo()
    {
        if (!$logo = self::getParams()->get("logo_file", null))
        {
            return;
        }

        return JURI::root() . $logo;
    }

	/**
	 *  Get website URL
	 *
	 *  @return  string  Site URL
	 */
	public static function getSiteURL()
	{
		return self::getParams()->get("sitename_url", JURI::root());
	}

    /**
     *  Get Plugin Parameters
     *
     *  @return  JRegistry
     */
    public static function getParams()
    {
    	if (self::$params)
		{
			return self::$params;
		}

		if (!$plugin = JPluginHelper::getPlugin('system', 'gsd'))
		{
			return;
		}

		self::$params = new JRegistry($plugin->params);
		return self::$params;
    }

}

?>