<?php

/**
 * @package         Google Structured Data Markup
 * @version         2.1.5 Free
 *
 * @author          Tassos Marinos <info@tassos.gr>
 * @link            http://www.tassos.gr
 * @copyright       Copyright © 2016 Tassos Marinos All Rights Reserved
 * @license         GNU GPLv3 <http://www.gnu.org/licenses/gpl.html> or later
 */

defined('_JEXEC') or die('Restricted access');

require_once JPATH_PLUGINS . '/system/nrframework/helpers/functions.php';
require_once __DIR__ . '/helper/helper.php';
require_once __DIR__ . "/helper/json.php";

class plgSystemGSD extends JPlugin
{
	/**
	 *  Auto loads the plugin language file
	 *
	 *  @var  boolean
	 */
	protected $autoloadLanguage = true;

	/**
	 *  The loaded indicator of helper
	 *
	 *  @var  boolean
	 */
	protected $init;

	/**
	 *  Application Object
	 *
	 *  @var  object
	 */
	protected $app;

	/**
	 *  JSON Helper
	 *
	 *  @var  class
	 */
	private $json;

	/**
	 *  Event to add custom script tags to the document
	 *
	 *  @return  boolean    True on success
	 */
	public function onBeforeRender()
	{
		// Load Helper
		if (!$this->getHelper())
		{
			return;
		}

		// Add Google Structured Markup
		$this->addGoogleStructuredMarkup();
	}

	/**
	 *  Adds Google Structured markup to the document in JSON Format
	 *
	 *  @return void
	 */
	private function addGoogleStructuredMarkup()
	{
		// Get JSON markup for each available type
		$data = array(
			$this->getJSONSiteName(),
			$this->getJSONSitelinksSearch(),
			
			$this->getJSONBreadcrumbs()
		);

		// Convert data array to string
		$markup = implode("\n", array_filter($data));

		// Return if markup is empty
		if (!$markup || empty($markup) || is_null($markup))
		{
			return;
		}

		// Add final markup to the document
		JFactory::getDocument()->addCustomTag('
            <!-- Start: ' . JText::_("GSD") . ' -->
            ' . $markup . '
            <!-- End: ' . JText::_("GSD") . ' -->
        ');
	}

	/**
	 *  Prepare form and add my field.
	 *
	 *  @param   JForm  $form  The form to be altered.
	 *  @param   mixed  $data  The associated data for the form.
	 *
	 *  @return  boolean
	 */
	function onContentPrepareForm($form, $data)
	{
		// Run only on backend
		if (!$this->app->isAdmin() || !$form instanceof JForm)
		{
			return;
		}

		switch ($form->getName())
		{
			case 'com_plugins.plugin':
				// Check we are manipulating the GSD plugin.
				if (!$form->getField('sitename_enabled', 'params'))
				{
					break;
				}

				GSDHelper::loadIntegrationsForms($form);
				break;

			
		}

		return true;
	}

	

	/**
	 *  Returns Breadcrumbs structured data markup
	 *  https://developers.google.com/structured-data/breadcrumbs
	 *
	 *  @return  string
	 */
	private function getJSONBreadcrumbs()
	{
		// Skip on homepage 
		if (!$this->params->get("breadcrumbs_enabled", true) || GSDHelper::isFrontPage())
		{
			return;
		}

		// Generate JSON
		return $this->json->setData(array(
			"snippet" => "breadcrumbs",
			"crumbs"  => GSDHelper::getCrumbs($this->params->get('breadcrumbs_home', JText::_('GSD_BREADCRUMBS_HOME')))
		))->generate();
	}

	/**
	 *  Returns Site Name strucuted data markup
	 *  https://developers.google.com/structured-data/site-name
	 *
	 *  @return  string on success, boolean on fail
	 */
	private function getJSONSiteName()
	{
		if (!$this->params->get("sitename_enabled", true) || !GSDHelper::isFrontPage())
		{
			return;
		}

		// Generate JSON
		return $this->json->setData(array(
			"snippet" => "sitename",
			"name"    => GSDHelper::getSiteName(),
			"url"     => GSDHelper::getSiteURL(),
			"alt"     => $this->params->get("sitename_name_alt")
		))->generate();
	}

	/**
	 *  Returns Sitelinks Searchbox structured data markup
	 *  https://developers.google.com/search/docs/data-types/sitelinks-searchbox
	 *
	 *  @return  string on success, boolean on fail
	 */
	private function getJSONSitelinksSearch()
	{
		if (!$sitelinks = $this->params->get('sitelinks_enabled', false))
		{
			return;
		}

		// Setup the right Search URL
		switch ($sitelinks)
		{
			case "1": // com_search
				$searchURL = GSDHelper::route(JURI::base() . 'index.php?option=com_search&searchphrase=all&searchword={search_term}');
				break;
			case "2": // com_finder
				$searchURL = GSDHelper::route(JURI::base() . 'index.php?option=com_finder&q={search_term}');
				break;
			case "3": // custom URL
				$searchURL = trim($this->params->get('sitelinks_search_custom_url'));
				break;
		}

		// Generate JSON
		return $this->json->setData(array(
			"snippet"   => "search",
			"siteurl"   => GSDHelper::getSiteURL(),
			"searchurl" => $searchURL
		))->generate();
	}

	

	/**
	 *  Loads Helper files
	 *
	 *  @return  boolean
	 */
	private function getHelper()
	{
		// Return if is helper is already loaded
		if ($this->init)
		{
			return true;
		}

		// Return if we are not in frontend
		if (!$this->app->isSite())
		{
			return false;
		}

		// Only on HTML documents
		if ((JFactory::getDocument()->getType() != 'html'))
		{
			return false;
		}

		// Return if current page is an XML page
		if (NRFrameworkFunctions::isFeed() || $this->app->input->getInt('print', 0))
		{
			return false;
		}

		// Initialize JSON Generator Class
		$this->json = new GSDJSON();

		return ($this->init = true);
	}
}
