<?php
/**
 * ConnectDaily Events Calendar Content Plugin 
 *  
 * This plugin implements the various connectDaily shortcodes 
 * that can be inserted into content. It also implements on the 
 * onAjax method for callbacks. 
 *  
 * Shortcodes are in the format: 
 *  
 * {connectdaily_xxxx param1="abc" param...n="def"} 
 *  
 * where xxxx is the specific shortcode value. E.G. simplelist 
 *  
 * @package Joomla.Plugin 
 * @subpackage Content.connectdaily 
 * @copyright (C) 2016 - MH Software, Inc. 
 * @license GNU/GPLv3 http://www.gnu.org/licenses/gpl-3.0.html 
 * @author gsexton (3/11/16) 
 */
defined('_JEXEC') or die;

jimport('joomla.plugin.plugin');
jimport('connectdaily.class-cdaily-plugin');
jimport('connectdaily.class-cdaily-joomla');
jimport('connectdaily.class-cdaily-eventsrenderer');
jimport('connectdaily.class-cdaily-datetime');
jimport('connectdaily.class-timedata');
jimport('connectdaily.detailed-list');
jimport('connectdaily.display-calendar');
jimport('connectdaily.CDailyAddEvent');
jimport('connectdaily.CDailySearch');
jimport('connectdaily.CDailySSO');

class plgContentConnectDaily extends JPlugin {

    /** @var CDailyJoomlaPlugin $plugin  */
    private $plugin;
    private $first = true;

    public function __construct(&$subject, $params) {
        parent::__construct($subject, $params);
        $this->plugin = CDailyJoomlaPlugin::getInstance();
    }

    /**
     * This is the method invoked by the com_ajax component to 
     * perform AJAX requests for connectDaily. The naming is kind 
     * of voodoo. 
     *  
     * Sample invocation URL: 
     *  
     * index.php?option=com_ajax&plugin=connectdaily&group=content&format=raw&action=cd_viewitem' 
     *  
     */
    public function onAjaxConnectdaily() {
        $plugin = $this->plugin;
        $app = JFactory::getApplication();
        $req = $app->input;

        $action = $req->getString('action', 'NONE');
        $result = '';

        switch ($action) {
        case 'cd_calendar':
            $result = $this->processCalendarNavigate($plugin, $req);
            break;
        case 'cd_daymap':
            $result = $this->processDayMap($plugin, $req);
            break;
        case 'cd_clearcache':
            $this->clearCache($plugin,$req);
            break;
        case 'cd_dismisshint':
            $plugin->markHintSeen($req->getString('hintName'));
            break;
        case 'cd_displayday':
            $result = $this->processDayEvents($plugin, $req);
            break;
        case 'cd_csshelper':
            $result=$this->processStyleCSS($plugin,$req,$app);
            break;
        case 'cd_icalendar':
            $result = $this->processiCalendar($plugin, $req);
            break;
        case 'cd_terms':
            $result=$this->getTermsAndConditions($plugin,$req);
            break;
        case 'cd_viewitem':
            $result = $this->processViewItem($plugin, $req);
            break;
        default:
            break;
        }
        return $result;
    }

    /**
     * Clear our local cache. Called by the cloud back-end. 
     *  
     * @param $plugin - Reference to Plugin object. 
     * @param $input - Joomla JInput Object 
     */
    private function clearCache($plugin,$input){
        if ($input->getMethod()==='POST') {
            $plugin->purgeTransients(true);
        }
    }

    private function getTermsAndConditions($plugin,$req){
        $sPath=JPATH_LIBRARIES.'/connectdaily/terms.html';
        return file_get_contents($sPath);
    }

    public function onContentPrepare($context, &$row, &$params, $page = 0) {
        if ($context == 'com_finder.indexer') {
            return true;
        } else if (strpos($row->text, '{connectdaily_') === false) {
            // Since a strpos is a lost less costly than a regex,
            // use it to eliminate a costly effort.
            return true;
        } else {
            $settings=$this->plugin->getSettings();
            if ($this->first) {
                if (!$settings->used) {
                    $this->plugin->markUsed();
                }
                $this->first = false;
                $this->plugin->addPluginScripts();
                $this->plugin->addPluginStylesheet();
            }
            return $this->processShortCodes($row->text);
        }
    }

    private function cleanArg($arg) {
        if (substr($arg, 0, 1) === '"' || substr($arg, 0, 1) === '\'') {
            $arg = substr($arg, 1);
        }
        $len = strlen($arg);
        if (substr($arg, $len - 1, 1) === '"' || substr($arg, $len - 1, 1) === '\'') {
            $arg = substr($arg, 0, $len - 1);
        }
        return $arg;
    }

    private function addDefaults(&$args, $defaults = null) {
        
        if ($defaults == null) {
            $defaults = array(
                'maxcount' => 16,
                'allow_duplicates' => 1,
                'dayspan' => 30,
                'datefmt' => $this->plugin->getSettings()->date_format,
                'other_options' => null
                );
        }
        foreach ($defaults as $key => $value) {
            if (!array_key_exists($key, $args) || (empty($args[$key]) && !$value === '0')) {
                $args[$key] = $value;
            }
        }
        return $args;
    }

    /**
     * Return the HTML for our events filtering short code.
     */
    private function doEventsFilter($tag, $args, &$text) {
        $renderer=new CDCalendarWriter($this->plugin);

        $result = $renderer->renderEventsFilter($args);

        $text = str_replace($tag, $result, $text);
    }

    /**
     * Render a single event.
     */
    private function doSingleEvent($tag, $args, &$text){
        $input=JFactory::getApplication()->input;
        $result='';
        $id=$input->getInt('cal_item_id',-1);
        if ( $id > 0 ){
            $args['by_id']=$id;
        }
        if (array_key_exists('by_id',$args)) {
            $args['by_method']='cal_item_id';
            if (!array_key_exists('id',$args)) {
                $args['id']='IDcdSingleItemList';
            }
            $lister=new CDEventLister($this->plugin);
            $result = $lister->detailedList($args,null,$tag).$this->plugin->ajaxURLScript();
        }
        $text = str_replace($tag, $result, $text);
    }

    /**
     * Write out the Ajax URL used for retrieving iCalendar.
     */
    private function doiCalendar($tag, $args, &$text) {
        $repl = $this->plugin->getAjaxURL('action=cd_icalendar&format=raw&by_method=' .
                                          $args['by_method'] . '&by_id=' . $args['by_id'],true);

        // TODO: This is broken. It looks like the Route call in getAjaxURL() is breaking things.

        $repl = 'webcal' . substr($repl, strpos($repl, ':'));
        $text = str_replace($tag, $repl, $text);
    }

    /**
      * Render the detailed list of events short code.
      */
    private function doDetailedList($tag, $args, &$text) {
        $renderer = new CDailyEventsRenderer($this->plugin);
        $id = array_key_exists('id', $args) ? $args['id'] : 'cdaily_detailedlist' . $this->plugin->getNextID();
        $content = $renderer->renderDetailedList($args, $id).$this->plugin->ajaxURLScript();
        $text = str_replace($tag, $content, $text);
    }



    private function doAddEvent($tag, $args, &$text) {

        $app = JFactory::getApplication();
        $input = $app->input;
        $renderer=new CDailyAddEvent($this->plugin);
        $output='';
        $bShowForm=true;
        $bRequiredPresent=true;
        $a=array();

       /*
            If this is a post, submit the form.
        */
        if ($input->getMethod()==='POST' && $input->getInt('CDcalendar_id',-1) >= 0) {
            $bShowForm=false;
            
            $fl=$renderer->getFieldsList();
            if (isset($args['required_fields'])) {
                /* 
                    There's a required_fields attribute set on the shortcode.
                 
                    Merge those into our field meta array for testing presence.
                */
                $fl=$renderer->mergeRequiredFields($fl,$args['required_fields']);
            }

            foreach ($fl as $field => $options){
                $fldName='CD'.$field;
                $val=$input->get($fldName,null,'raw');                
                if ($val===null || $val==="") {
                    if ($options['required']===true) {
                        $bRequiredPresent=false;
                    }
                    continue;
                }

                $val='';
                switch ($options['type']) {
                case 'intset':
                    $intvalues=$input->get($fldName,array(),'array');
                    if (gettype($intvalues)=='array') {
                        foreach ($intvalues as $intvalue ) {    
                            $val.=intval($intvalue,10).',';
                        }
                    } else {
                        if (!empty($intvalues)) {
                            $val=$intvalues;
                        }
                    }
                    $val=trim($val,',');
                    break;
                case 'int':
                    $val=$input->getInt($fldName,0);
                    break;
                default:
                    $val=trim($input->getString($fldName));
                    break;
                }
                $a[$field]=$val;
            }
            if ($bRequiredPresent) {
                $a["X-Forwarded-For"] = JFactory::getApplication()->input->server->get('REMOTE_ADDR');
                $result=$renderer->processSubmit($a);
                $s=$renderer->getErrors($result);
                if ($s!=null) {
                    $bShowForm=true;
                    $app->enqueueMessage($s,'error');
                }
                $s=$renderer->getConflicts($result);
                if ($s!=null) {
                    $bShowForm=true;
                    $app->enqueueMessage($s,'error');
                    $output.='<div class="error cdaily-error">'.$s.'</div>';
                }
                $s=$renderer->getWarnings($result);
                if ($s!=null) {
                    $app->enqueueMessage($s,'warning');
                }
            } else {
                $bShowForm=true;
                $app->enqueueMessage($this->plugin->translate('COM_CONNECTDAILY_REQFLDNOTSUPPLIED'),'error');
            }
        } else {
            $this->addDefaults($args, array(
                'allow_recurrence' => '1',
                'id' => 'cdaily-addevent'
                ));
        }

        if ($bShowForm) {
            $output.=$renderer->addEventForm($args,$a);
        } else {
            $output.='<div class="success">'.$renderer->getSuccessOutput().'</div>';
        }

        $text = str_replace($tag, $output, $text);
    }

    /**
     * We have some views that aren't available as native Joomla 
     * short codes. Since they're really not commonly used, give 
     * users the option to get them vi an IFRAME tag. 
     */
    private function doCalendarIFrame($tag, $args, &$text) {
        $this->addDefaults($args, array(
                               'id' => 'cdaily-calendariframe' . $this->plugin->getNextID(),
                               'height' => '1024px',
                               'width' => '600px',
                               'view' => 'View.html'
                               ));
        $result = '<iframe id="' . $args['id'] . '" scrolling="auto" style="height:
        ' . $args['height'] . '; width: ' . $args['width'] . ';" ' .
            'src="' . $this->plugin->getSettings()->url .
            $args['view'] . '?' . $args['by_method'] . '=' . $args['by_id'] . '"></iframe>';
        $text = str_replace($tag, $result, $text);
    }

    private function doMiniCalendar($tag, $args, &$text) {
        $renderer = new CDCalendarWriter($this->plugin);
        $text = str_replace($tag, $renderer->renderMiniCalendar($args).$this->plugin->ajaxURLScript(), $text);
    }

    /**
     * Render the month view short code.
     */
    private function doMonthView($tag, $args, &$text) {
        
        $renderer = new CDCalendarWriter($this->plugin);
        $text = str_replace($tag, $renderer->renderMonth($args, false), $text);
    }

    /** 
     * Draw our search calendar form, and if appropriate, execute 
     * the search. 
     */
    private function doSearch($tag, $args, &$text) {
        $app = JFactory::getApplication();
        $input = $app->input;
        $renderer = new CDailySearch($this->plugin);
        $id = array_key_exists('id', $args) ? $args['id'] : 'cdaily_search' . $this->plugin->getNextID();
        $content = $renderer->renderSearchInput($args).$this->plugin->ajaxURLScript();
        $searchVal=$input->getString(CDailySearch::SEARCH_FIELD_NAME);

        if (!empty($searchVal)) {
            $args[CDailySearch::SEARCH_FIELD_NAME]=stripslashes($searchVal);
            $content.=$renderer->executeSearch($args, $id);
        }

        $text = str_replace($tag, $content, $text);
    }

    /**
     * Render the Simple List of Events Short Code.
     */
    private function doSimpleList($tag, $args, &$text) {
        $renderer = new CDailyEventsRenderer($this->plugin);
        $id = array_key_exists('id', $args) ? $args['id'] : 'cdaily_simplelist' . $this->plugin->getNextID();
        $content = $renderer->renderSimpleList($args, $id).$this->plugin->ajaxURLScript();
        $text = str_replace($tag, $content, $text);
    }



    /**
     * Given a shortcode tag, split the tag up and return the args 
     * as an associative array 
     *  
     * @param $tag string 
     *  
     * @return array 
     */
    private function getArguments($tag) {
        $result = array();
        $tag = substr($tag, 1, strlen($tag) - 2);
        $parts = preg_split('/\s+/', $tag);
        foreach ($parts as $part) {
            if (strpos($part, '=') > 0) {
                if (strpos($part,'other_options=')===0) {
                    $val=$this->cleanArg(substr($part,strpos($part,'=')+1));
                    $opts=preg_split('/&/',$val);
                    foreach ($opts as $oopt){
                        $halves = preg_split('/=/', trim($oopt));
                        $result[trim($halves[0])] = $this->cleanArg(trim($halves[1]));
                    }
                } else {
                    $halves = preg_split('/=/', trim($part));
                    $result[trim($halves[0])] = $this->cleanArg(trim($halves[1]));
                }
            }
        }
        
        return $result;
    }


    private function processCalendarNavigate($plugin, $req) {
        $cal = new CDCalendarWriter($plugin);
        $args = $req->getArray(array(
                                   'by_method' => 'string',
                                   'by_id' => 'int',
                                   'calendar_id' => 'string',
                                   'item_type_id' => 'string',
                                   'resource_id' => 'string',
                                   'reource_type_id' => 'string',
                                   'location_id' => 'string',
                                   'year' => 'int',
                                   'month' => 'int',
                                   'offset' => 'int',
                                   'wrap_events' => 'int',
                                   'enable_dropdown' => 'int',
                                   'enable_styles' => 'int'
                                   ));

        return $cal->renderMonth($args, true);
    }

    private function processDayMap($plugin, $req) {
        $args = $req->getArray(array(
                                   'date' => 'int',
                                   'by_method' => 'string',
                                   'by_id' => 'int',
                                   'calendar_id' => 'string',
                                   'item_type_id' => 'string',
                                   'resource_id' => 'string',
                                   'reource_type_id' => 'string',
                                   'location_id' => 'string'
                                   ));
        $args['start'] = $args['date'];
        $json_url = 'jsonp/' . $args['by_method'] . '/daymap/' . $args['by_id'] . '.js';
        $reqData = $plugin->getPostData('daymap', $json_url, $args);
        if (!$reqData->wasSuccess()) {
            $reqData->content = '{}';
        }

        return $req->get('callback', '') . '(' . $reqData->content . ');';
    }

    /**
     * Process an AJAX request for viewing a specific day. Called by 
     * the month view calendars. 
     */
    private function processDayEvents($plugin, $req) {
        $fields = $req->getArray(array(
                                'by_method' => 'int',
                                'by_id' => 'int',
                                'calendar_id' => 'string',
                                'item_type_id' => 'string',
                                'resource_id' => 'string',
                                'reource_type_id' => 'string',
                                'location_id' => 'string',
                                'render_link' => 'int',
                                'date' => 'int',
                                'yr' => 'int',
                                'mo' => 'int',
                                'da' => 'int'
                                ));
        $fields = $req->getArray($fields);
        $fields['maxcount'] = 0;
        $fields['dayspan'] = 0;
        $fields['start'] = $fields['date'];
        $renderer = new CDCalendarWriter($plugin);
        return $renderer->renderSpecificDay($fields);
    }

    /**
     * Process an AJAX request for the Event Style CSS file. This is 
     * the by event type, or by style id information. 
     */
    private function processStyleCSS($plugin, $req, $app) {

        /* 
            OK, the next 3 lines are pretty much voodoo. If you
            don't set the mime type, then a call to setHeader('content-type') fails.
         
            If you don't allow cache, then the expires and cache-control
            headers can't be set.
        */
        $doc=JFactory::getDocument();
        $doc->setMimeEncoding("text/css"); 
        $app->allowCache(true);

        $dt=new DateTime("now",new DateTimeZone("UTC"));
        $dt=$dt->add(new DateInterval('PT30M'));

        $app->setHeader('Expires',$dt->format(DateTime::RFC1123),true);
        $app->setHeader('Cache-Control','max-age=1800',true);

        $renderer=new CDCalendarWriter($plugin);

        return $renderer->renderStyleCSS();
    }

    /**
     * Perform the actual iCalendar Request and return the file.
     */
    private function processiCalendar($plugin, $req) {
        /*
            Get the file.
        */
        $by_method = $req->getString('by_method', 'calendar_id');
        $by_id = $req->getInt('by_id', 0);
        $args = array(
            'nodefaultcontact' => '1',
            'show_resources' => 0
            );
        $doc = JFactory::getDocument();
        $file = $plugin->getPostData('icalendar', 'iCal/' . $by_method . '/' . $by_id . '.ics', $args);
        $doc->setMimeEncoding('text/calendar');
        return $file->content;
    }

    private function processShortCodes(&$text) {
        $result = false;
        $matches = array();
        while (preg_match("/\{connectdaily_([a-z_]+)[^\}]*\}/", $text, $matches) == 1) {
            $result = true;

            $args = $this->getArguments($matches[0]);

            switch ($matches[1]) {
            case 'addevent':
                $this->doAddEvent($matches[0], $args, $text);
                break;
            case 'calendariframe':
                $this->doCalendarIFrame($matches[0], $args, $text);
                break;
            case 'detailedlist':
                $this->doDetailedList($matches[0], $this->addDefaults($args), $text);
                break;
            case 'event':
                $this->doSingleEvent($matches[0], $args,$text);
                break;
            case 'filter':
                $this->doEventsFilter($matches[0], $args, $text);
                break;
            case 'icalendar':
                $this->doiCalendar($matches[0], $args, $text);
                break;
            case 'minicalendar':
                $this->doMiniCalendar($matches[0], $args, $text);
                break;
            case 'monthview':
                $this->addDefaults($args);
                $this->doMonthView($matches[0], $args, $text);
                break;
            case 'search':
                $this->addDefaults($args);
                $this->doSearch($matches[0],$args,$text);
                break;
            case 'simplelist':
                $this->doSimpleList($matches[0], $this->addDefaults($args), $text);
                break;
            case 'test':
                $text = str_replace($matches[0], "This is Test Content from connectDaily Plugin", $text);
                break;
            default:
                $err = "connectDaily Plugin - Unhandled value of matches=$matches[1]";
                JFactory::getApplication()->enqueueMessage($err, 'error');
                $this->plugin->logError($err);
                return false;
                break;
            }
        }
        return $result;
    }

    /**
     * Process an AJAX request for viewing a specific item.
     */
    private function processViewItem($plugin, $req) {
        $fields = array(
            'cal_item_id' => 'int',
            'date' => 'int',
            'allow_duplicates' => 'int',
            'show_resources' => 'int'
            );

        $lister = new CDailyEventsRenderer($plugin);
        $fields = $req->getArray($fields);
        $lister->showEnds=true;
        return $lister->processViewItem($fields);
    }
}
